/******************************************************************************
 * Copyright (C) Siarhei Uzunbajakau, 2023.
 *
 * This program is free software. You can use, modify, and redistribute it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 or (at your option) any later version.
 * This program is distributed without any warranty.
 *
 * Refer to COPYING.LESSER for more details.
 ******************************************************************************/

#ifndef SolverMMSV_H__
#define SolverMMSV_H__

#include <deal.II/grid/grid_in.h>
#include <deal.II/grid/grid_out.h>
#include <deal.II/grid/grid_tools.h>
#include <deal.II/grid/manifold_lib.h>

#include <string>

#include "exact_solution.hpp"
#include "settings.hpp"
#include "static_vector_solver_i.hpp"

using namespace StaticVectorSolver;

/**
 * \brief Implements the *Method of manufactured solutions, vector potential*
 * [(mms-v/)](@ref page_mms_v) numerical experiment.
 *****************************************************************************/
template<int dim>
class SolverMMSV
  : public SettingsMMSV
  , public Solver1<dim>
{
public:
  SolverMMSV() = delete;

  /**
   * The constructor.
   *
   * @param[in] p - The degree of the Nedelec finite elements.
   * @param[in] mapping_degree - The degree of the interpolating polynomials
   * used for mapping.
   * @param[in] r - The parameter that encodes the degree of mesh refinement.
   * Must coincide with one of the values set in mms-v/gmsh/build. This
   * parameter is used to compose the name of the mesh file to be uploaded
   * from mms-v/gmsh/data/.
   * @param[in] fname - The name of the vtu file without extension to save
   * the data.
   *****************************************************************************/
  SolverMMSV(unsigned int p,
             unsigned int mapping_degree,
             unsigned int r,
             std::string fname)
    : Solver1<dim>(p,
                   mapping_degree,
                   3,
                   0.0,
                   fname,
                   &exact_solution,
                   print_time_tables,
                   project_exact_solution,
                   true)
    , r(r)
    , fname(fname)
  {
    if (DIMENSION__ == 2) {
      if (HYPERCUBE__ == 1) {
        fname_mesh = "../../gmsh/data/square_r" + std::to_string(r) + ".msh";
      } else {
        fname_mesh = "../../gmsh/data/circle_r" + std::to_string(r) + ".msh";
      }
    } else {
      if (HYPERCUBE__ == 1) {
        fname_mesh = "../../gmsh/data/cube_r" + std::to_string(r) + ".msh";
      } else {
        fname_mesh = "../../gmsh/data/sphere_r" + std::to_string(r) + ".msh";
      }
    }

    Solver1<dim>::run();
  }

  ~SolverMMSV() = default;

private:
  const unsigned int r;
  const std::string fname;
  std::string fname_mesh;

  const ExactSolutionMMSV_A<dim> exact_solution;

  virtual void make_mesh() override final;
  virtual void fill_dirichlet_stack() override final;
  virtual void solve() override final;

  const SphericalManifold<dim> sphere;
};

template<int dim>
void
SolverMMSV<dim>::fill_dirichlet_stack()
{
  Solver1<dim>::dirichlet_stack = { { bid_dirichlet, &exact_solution } };
}

template<int dim>
void
SolverMMSV<dim>::solve()
{
  SolverControl control(1000 * Solver1<dim>::system_rhs.size(),
                        1e-6 * Solver1<dim>::system_rhs.l2_norm(),
                        false,
                        false);

  if (log_cg_convergence)
    control.enable_history_data();

  GrowingVectorMemory<Vector<double>> memory;
  SolverCG<Vector<double>> cg(control, memory);

  //  PreconditionJacobi<SparseMatrix<double>> preconditioner;
  //  preconditioner.initialize(Solver1<dim>::system_matrix, 1.0);

  PreconditionSSOR<SparseMatrix<double>> preconditioner;
  preconditioner.initialize(Solver1<dim>::system_matrix, 1.2);

  cg.solve(Solver1<dim>::system_matrix,
           Solver1<dim>::solution,
           Solver1<dim>::system_rhs,
           preconditioner);

  Solver1<dim>::constraints.distribute(Solver1<dim>::solution);

  if (log_cg_convergence) {
    const std::vector<double> history_data = control.get_history_data();

    std::ofstream ofs(fname + "_cg_convergence.csv");

    unsigned int i = 1;
    for (auto item : history_data) {
      ofs << i << ", " << item << "\n";
      i++;
    }
    ofs.close();
  }
}

template<int dim>
void
SolverMMSV<dim>::make_mesh()
{
  GridIn<dim> gridin;
  gridin.attach_triangulation(Solver1<dim>::triangulation);

  std::ifstream ifs(fname_mesh);
  gridin.read_msh(ifs);

  Solver1<dim>::triangulation.reset_all_manifolds();

  if (HYPERCUBE__ != 1) {
    for (auto cell : Solver1<dim>::triangulation.active_cell_iterators()) {

      for (unsigned int f = 0; f < GeometryInfo<dim>::faces_per_cell; f++) {

        double dif_norm = 0.0;
        for (unsigned int v = 1; v < GeometryInfo<dim>::vertices_per_face; v++)
          dif_norm += std::abs(cell->face(f)->vertex(0).norm() -
                               cell->face(f)->vertex(v).norm());

        if ((dif_norm < eps) && (cell->center().norm() > rd1))
          cell->face(f)->set_all_manifold_ids(1);
      }
    }
  }

  Solver1<dim>::triangulation.set_manifold(1, sphere);
}
#endif
